<?php

namespace WP_Defender\Model\Notification;

use WP_Defender\Model\Scan;
use WP_Defender\Model\Scan_Item;
use WP_Defender\Traits\IO;

/**
 * Class Firewall_Notification
 * @package WP_Defender\Model\Notification
 */
class Malware_Notification extends \WP_Defender\Model\Notification {
	use IO;

	/**
	 * Option name.
	 * @var string
	 */
	protected $table = 'wd_malware_scanning_notification';

	public function before_load() {
		$default = array(
			'slug'                 => 'malware-notification',
			'title'                => __( 'Malware Scanning - Notification', 'wpdef' ),
			'status'               => self::STATUS_DISABLED,
			'description'          => __( 'Get email notifications when Defender has finished manual malware scans.', 'wpdef' ),
			'in_house_recipients'  => array(
				$this->get_default_user(),
			),
			'out_house_recipients' => array(),
			'type'                 => 'notification',
			'dry_run'              => false,
			'configs'              => array(
				'always_send' => false,
				'error_send'  => false,
				'template'    => array(
					'found'     => array(
						'subject' => __( 'Scan of {SITE_URL} complete. {ISSUES_COUNT} issues found.', 'wpdef' ),
						'body'    => __(
							'Hi {USER_NAME},

WP Defender here, reporting back from the front.

I\'ve finished scanning {SITE_URL} for vulnerabilities and I found {ISSUES_COUNT} issues that you should take a closer look at!
{ISSUES_LIST}

Stay Safe,
WP Defender
WPMU DEV Superhero
',
							'wpdef'
						),
					),
					'not_found' => array(
						'subject' => __( 'Scan of {SITE_URL} complete. {ISSUES_COUNT} issues found.', 'wpdef' ),
						'body'    => __(
							'Hi {USER_NAME},

WP Defender here, reporting back from the front.

I\'ve finished scanning {SITE_URL} for vulnerabilities and I found nothing. Well done for running such a tight ship!

		Keep up the good work! With regular security scans and a well-hardened installation you\'ll be just fine.

Stay safe,
WP Defender
WPMU DEV Superhero
',
							'wpdef'
						),
					),
					'error'     => array(
						'subject' => __( 'Couldn’t scan {SITE_URL} for vulnerabilities. ', 'wpdef' ),
						'body'    => __(
							'Hi {USER_NAME},

WP Defender here, reporting back from the front.

I couldn’t scan {SITE_URL} for vulnerabilities. Please {follow this link} and check the logs to see what casued the failure.

Stay Safe,
WP Defender
WPMU DEV Superhero
',
							'wpdef'
						),
					),
				),
			),
		);
		$this->import( $default );
	}

	/**
	 * @param Scan $scan
	 *
	 * @return void
	 */
	public function send( Scan $scan ) {
		$object = $this;
		if ( $scan->is_automation ) {
			$object = wd_di()->get( Malware_Report::class );
			if ( true === $object->dry_run ) {
				// No send because the report option without recipients is enabled.
				return;
			}
		}
		$issues = $scan->get_issues( null, Scan_Item::STATUS_ACTIVE );
		// Sometimes value may be as empty string for disabled 'always_send'.
		if ( empty( $object->configs['always_send'] ) && 0 === count( $issues ) ) {
			return;
		}
		$templates = count( $issues ) ? $object->configs['template']['found'] : $object->configs['template']['not_found'];
		$subject   = $templates['subject'];
		$content   = nl2br( $templates['body'] );
		$slug      = $scan->is_automation ? '#malware_report' : '#malware-notification';
		// Value $slug is required to have various module popups on the Notifications page.
		$report_url = apply_filters( 'wp_defender/scan/email_report_link', network_admin_url( 'admin.php?page=wdf-notification' . $slug ) );
		foreach ( $object->in_house_recipients as $user ) {
			if ( self::USER_SUBSCRIBED !== $user['status'] ) {
				continue;
			}
			$this->send_to_user( $user['email'], $user['name'], $subject, $content, $issues, $report_url );
		}
		foreach ( $object->out_house_recipients as $user ) {
			if ( self::USER_SUBSCRIBED !== $user['status'] ) {
				continue;
			}
			$this->send_to_user( $user['email'], $user['name'], $subject, $content, $issues, $report_url );
		}
	}

	/**
	 * @param $email
	 * @param $name
	 * @param $subject
	 * @param $content
	 * @param $issues
	 * @param $report_url
	 *
	 * @throws \DI\DependencyException
	 * @throws \DI\NotFoundException
	 */
	private function send_to_user( $email, $name, $subject, $content, $issues, $report_url ) {
		$controller_scan = wd_di()->get( \WP_Defender\Controller\Scan::class );
		$replacers       = array(
			'issues_count' => count( $issues ),
			'issues_list'  => $controller_scan->render_partial(
				'email/scan-issue-list',
				array(
					'issues' => $issues,
				),
				false
			),
			'site_url'     => network_site_url(),
			'user_name'    => $name,
		);
		// Not use the filter 'report_email_logs_link' because SCAN_PAGE_LINK doesn't exist.
		$replacers   = apply_filters( 'wd_notification_email_params', $replacers );
		$filter_name = count( $issues ) > 0
			? 'wd_notification_email_subject_issue'
			: 'wd_notification_email_subject';
		$subject     = apply_filters( $filter_name, $subject );
		$content     = apply_filters( 'wd_notification_email_content_before', $content );
		foreach ( $replacers as $key => $replacer ) {
			$content = str_replace( "{{$key}}", $replacer, $content );
			$content = str_replace( '{' . strtoupper( $key ) . '}', $replacer, $content );
			$subject = str_replace( "{{$key}}", $replacer, $subject );
			$subject = str_replace( '{' . strtoupper( $key ) . '}', $replacer, $subject );
		}
		$content  = apply_filters( 'wd_notification_email_content_after', $content );
		$template = $controller_scan->render_partial(
			'email/scan',
			array(
				'subject'    => $subject,
				'content'    => $content,
				'report_url' => $report_url,
			),
			false
		);

		$headers = defender_noreply_html_header(
			defender_noreply_email( 'wd_scan_noreply_email' )
		);

		$ret = wp_mail( $email, $subject, $template, $headers );
		if ( $ret ) {
			$this->save_log( $email );
		}
	}

	/**
	 * Define labels for settings key.
	 *
	 * @param  string|null $key
	 *
	 * @return string|array|null
	 */
	public function labels( $key = null ) {
		$labels = array(
			'notification'                  => __( 'Malware Scanning - Notification', 'wpdef' ),
			'always_send_notification'      => __( 'Send notifications when no issues are detected', 'wpdef' ),
			'error_send'                    => __( "Send notifications when Defender couldn't scan the files", 'wpdef' ),
			'email_subject_issue_found'     => __( 'Email subject when an issue is found', 'wpdef' ),
			'email_subject_issue_not_found' => __( 'Email subject when no issues are found', 'wpdef' ),
			'email_subject_error'           => __( 'Email subject when failed to scan', 'wpdef' ),
			'email_content_issue_found'     => __( 'Email body when an issue is found', 'wpdef' ),
			'email_content_issue_not_found' => __( 'Email body when no issues are found', 'wpdef' ),
			'email_content_error'           => __( 'When failed to scan', 'wpdef' ),
			'notification_subscribers'      => __( 'Recipients', 'wpdef' ),
		);

		if ( ! is_null( $key ) ) {
			return isset( $labels[ $key ] ) ? $labels[ $key ] : null;
		}

		return $labels;
	}

	/**
	 * Try to send email if:
	 * Malware Scanning - Notification is enabled,
	 * Malware Scanning - Reporting is enabled or Scan model has is_automation = true.
	 *
	 * @return bool
	*/
	public function check_options() {
		$malware_report = new \WP_Defender\Model\Notification\Malware_Report();
		return self::STATUS_ACTIVE === $this->status || self::STATUS_ACTIVE === $malware_report->status;
	}
}