<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Slider block
 *
 * @since 1.0.0
 */
class aThemes_Slider extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve icon list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'athemes-hero-slider';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve icon list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'aThemes: Slider', 'airi' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve icon list widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-slider-push';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon list widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'airi-elements' ];
	}

	/**
	 * Register icon list widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls() {
		$this->start_controls_section(
			'section_icon',
			[
				'label' => __( 'Slides', 'airi' ),
			]
		);

		$this->add_control(
			'slides_control',
			[
				'label' => '',
				'type' => Controls_Manager::REPEATER,
				'default' => [
					[
						'bg_image' 		=> [
							'url' => Utils::get_placeholder_image_src(),
						],							
						'slide_title' 	=> __( 'Welcome to our website', 'airi' ),
						'slide_text' 	=> 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Sed consectetur leo ac euismod commodo. Vivamus suscipit odio eget felis lacinia hendrerit sit amet non tortor',
						'linktext' 		=> '',
						'link'			=> '',
					],
					[
						'bg_image' 		=> [
							'url' => Utils::get_placeholder_image_src(),
						],						
						'slide_title' 	=> __( 'We hope you enjoy your stay', 'airi' ),
						'slide_text' 	=> 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Sed consectetur leo ac euismod commodo. Vivamus suscipit odio eget felis lacinia hendrerit sit amet non tortor',
						'linktext' 		=> '',
						'link'			=> '',
					],
				],
				'fields' => [
					[
						'name' => 'bg_image',
						'label' => __( 'Background image', 'airi' ),
						'type' => Controls_Manager::MEDIA,
						'label_block' => true,
						'placeholder' => '',
						'default' => [
							'url' => Utils::get_placeholder_image_src(),
						],					
					],							
					[
						'name' => 'slide_title',
						'label' => __( 'Slide title', 'airi' ),
						'type' => Controls_Manager::TEXT,
						'label_block' => true,
						'placeholder' => __( 'Welcome to our website', 'airi' ),
						'default'	  => __( 'Welcome to our website', 'airi' ),
					],					
					[
						'name' => 'slide_text',
						'label' => __( 'Slide text', 'airi' ),
						'type' => Controls_Manager::TEXTAREA,
						'label_block' => true,
						'default' => 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Sed consectetur leo ac euismod commodo. Vivamus suscipit odio eget felis lacinia hendrerit sit amet non tortor',
					],
					[
						'name' => 'linktext',
						'label' => __( 'Button text', 'airi' ),
						'type' => Controls_Manager::TEXT,
						'label_block' => true,
						'placeholder' => __( 'Read more', 'airi' ),
						'default'	  => '',
					],
					[
						'name' => 'link',
						'label' => __( 'Button link', 'airi' ),
						'type' => Controls_Manager::URL,
						'label_block' => true,
						'placeholder' => __( 'https://example.com', 'airi' ),
					],
				],
				'title_field' => '{{{ slide_title }}}',
			]
		);

		$this->add_control(
			'view',
			[
				'label' => __( 'View', 'airi' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->add_responsive_control(
			'slider_height',
			[
				'label' => __( 'Slider height', 'airi' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 500,
				],
				'range' => [
					'px' => [
						'min' => 400,
						'max' => 1000
					],
				],
				'selectors' => [
					'{{WRAPPER}} .athemes-hero-wrapper .slick-slide' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_title_style',
			[
				'label' => __( 'Title', 'airi' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => __( 'Text Color', 'airi' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .athemes-hero-wrapper .slide-title' => 'color: {{VALUE}};',
				],
				'scheme' => [
					'type' => Scheme_Color::get_type(),
					'value' => Scheme_Color::COLOR_2,
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'selector' => '{{WRAPPER}} .athemes-hero-wrapper .slide-title',
				'scheme' => Scheme_Typography::TYPOGRAPHY_3,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_text_style',
			[
				'label' => __( 'Text', 'airi' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'text_color',
			[
				'label' => __( 'Text Color', 'airi' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .athemes-hero-wrapper .slide-text' => 'color: {{VALUE}};',
				],
				'scheme' => [
					'type' => Scheme_Color::get_type(),
					'value' => Scheme_Color::COLOR_2,
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'text_typography',
				'selector' => '{{WRAPPER}} .athemes-hero-wrapper .slide-text',
				'scheme' => Scheme_Typography::TYPOGRAPHY_3,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_button_style',
			[
				'label' => __( 'Button', 'airi' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'typography',
				'scheme' => Scheme_Typography::TYPOGRAPHY_4,
				'selector' => '{{WRAPPER}} a.button, {{WRAPPER}} a.button',
			]
		);

		$this->start_controls_tabs( 'tabs_button_style' );

		$this->start_controls_tab(
			'tab_button_normal',
			[
				'label' => __( 'Normal', 'elementor' ),
			]
		);

		$this->add_control(
			'button_text_color',
			[
				'label' => __( 'Text Color', 'elementor' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} a.button, {{WRAPPER}} .elementor-button' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'background_color',
			[
				'label' => __( 'Background Color', 'elementor' ),
				'type' => Controls_Manager::COLOR,
				'scheme' => [
					'type' => Scheme_Color::get_type(),
					'value' => Scheme_Color::COLOR_4,
				],
				'selectors' => [
					'{{WRAPPER}} a.button, {{WRAPPER}} .elementor-button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_button_hover',
			[
				'label' => __( 'Hover', 'elementor' ),
			]
		);

		$this->add_control(
			'hover_color',
			[
				'label' => __( 'Text Color', 'elementor' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a.button:hover, {{WRAPPER}} .elementor-button:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'button_background_hover_color',
			[
				'label' => __( 'Background Color', 'elementor' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a.button:hover, {{WRAPPER}} .elementor-button:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'button_hover_border_color',
			[
				'label' => __( 'Border Color', 'elementor' ),
				'type' => Controls_Manager::COLOR,
				'condition' => [
					'border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} a.button:hover, {{WRAPPER}} .elementor-button:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'hover_animation',
			[
				'label' => __( 'Hover Animation', 'elementor' ),
				'type' => Controls_Manager::HOVER_ANIMATION,
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'border',
				'placeholder' => '1px',
				'default' => '1px',
				'selector' => '{{WRAPPER}} .button',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'border_radius',
			[
				'label' => __( 'Border Radius', 'elementor' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} a.button, {{WRAPPER}} .elementor-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'button_box_shadow',
				'selector' => '{{WRAPPER}} .elementor-button',
			]
		);

		$this->add_responsive_control(
			'text_padding',
			[
				'label' => __( 'Padding', 'elementor' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'{{WRAPPER}} a.button, {{WRAPPER}} .elementor-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator' => 'before',
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Render icon list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings();
		$total = count( $settings['slides_control'] );

		?>
		<div class="athemes-hero-wrapper">
			<ul class="elementor-slick-slider athemes-slick-slider athemes-hero-slider">
				<?php foreach ( $settings['slides_control'] as $index => $item ) :
					$repeater_setting_key = $this->get_repeater_setting_key( 'text', 'slides_control', $index );

					$this->add_render_attribute( $repeater_setting_key, 'class', 'elementor-icon-list-text' );

					$this->add_inline_editing_attributes( $repeater_setting_key );
					?>
					<li style="background-image: url('<?php echo $item['bg_image']['url']; ?>');">
						<div class="slide-inner">
							<div class="container">
								<div class="row">
									<div class="col-md-6">
										<h2 class="slide-title"><?php echo esc_attr( $item['slide_title'] ); ?></h2>
										<p class="slide-text"><?php echo esc_attr( $item['slide_text'] ); ?></p>

										<?php
										if ( ! empty( $item['link']['url'] ) ) {
											$link_key = 'link_' . $index;

											$this->add_render_attribute( $link_key, 'href', $item['link']['url'] );

											if ( $item['link']['is_external'] ) {
												$this->add_render_attribute( $link_key, 'target', '_blank' );
											}

											if ( $item['link']['nofollow'] ) {
												$this->add_render_attribute( $link_key, 'rel', 'nofollow' );
											}

											echo '<a class="readmore button" ' . $this->get_render_attribute_string( $link_key ) . '>';
										}
										echo esc_html( $item['linktext'] );
										if ( ! empty( $item['link']['url'] ) ) {
											echo '</a>';
										}
										?>
									</div>
								</div>
							</div>
						</div>
					</li>
					<?php
				endforeach;
				?>
			</ul>	
		</div>
		<?php
	}

	/**
	 * Render icon list widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _content_template() {
	}
}


Plugin::instance()->widgets_manager->register_widget_type( new aThemes_Slider() );